import { EditorTheme } from "@kie-tools-core/editor/dist/api";
import { OperatingSystem } from "@kie-tools-core/operating-system";
import { editor, KeyCode, KeyMod } from "@kie-tools-core/monaco-editor";
export var MonacoEditorOperation;
(function (MonacoEditorOperation) {
    MonacoEditorOperation[MonacoEditorOperation["UNDO"] = 0] = "UNDO";
    MonacoEditorOperation[MonacoEditorOperation["REDO"] = 1] = "REDO";
    MonacoEditorOperation[MonacoEditorOperation["EDIT"] = 2] = "EDIT";
})(MonacoEditorOperation || (MonacoEditorOperation = {}));
export class MonacoEditorController {
    constructor(content, onContentChange, language, operatingSystem, isReadOnly, setValidationErrors) {
        this.onContentChange = onContentChange;
        this.language = language;
        this.operatingSystem = operatingSystem;
        this.isReadOnly = isReadOnly;
        this.setValidationErrors = setValidationErrors;
        this.getValidationMarkers = () => {
            return editor.getModelMarkers({});
        };
        this.model = editor.createModel(content, this.language);
        this.model.onDidChangeContent((event) => {
            var _a;
            if (!event.isUndoing && !event.isRedoing) {
                (_a = this.editor) === null || _a === void 0 ? void 0 : _a.pushUndoStop();
                onContentChange(this.model.getValue(), MonacoEditorOperation.EDIT);
            }
        });
        editor.onDidChangeMarkers(() => {
            this.setValidationErrors(this.getValidationMarkers());
        });
    }
    redo() {
        var _a, _b;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.focus();
        (_b = this.editor) === null || _b === void 0 ? void 0 : _b.trigger("editor", "redo", null);
    }
    undo() {
        var _a, _b;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.focus();
        (_b = this.editor) === null || _b === void 0 ? void 0 : _b.trigger("editor", "undo", null);
    }
    setTheme(theme) {
        editor.setTheme(this.getMonacoThemeByEditorTheme(theme));
    }
    show(container, theme) {
        if (!container) {
            throw new Error("We need a container to show the editor!");
        }
        if (this.editor !== undefined) {
            this.setTheme(theme);
            return this.editor;
        }
        this.editor = editor.create(container, {
            model: this.model,
            language: this.language,
            scrollBeyondLastLine: false,
            automaticLayout: true,
            fontSize: 12,
            theme: this.getMonacoThemeByEditorTheme(theme),
            readOnly: this.isReadOnly,
        });
        this.editor.addCommand(KeyMod.CtrlCmd | KeyCode.KeyZ, () => {
            this.onContentChange(this.model.getValue(), MonacoEditorOperation.UNDO);
        });
        this.editor.addCommand(KeyMod.CtrlCmd | KeyMod.Shift | KeyCode.KeyZ, () => {
            this.onContentChange(this.model.getValue(), MonacoEditorOperation.REDO);
        });
        if (this.operatingSystem !== OperatingSystem.MACOS) {
            this.editor.addCommand(KeyMod.CtrlCmd | KeyCode.KeyY, () => {
                this.onContentChange(this.model.getValue(), MonacoEditorOperation.REDO);
            });
        }
        return this.editor;
    }
    getContent() {
        var _a, _b;
        return ((_b = (_a = this.editor) === null || _a === void 0 ? void 0 : _a.getModel()) === null || _b === void 0 ? void 0 : _b.getValue()) || "";
    }
    forceRedraw() {
        var _a;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.render(true);
    }
    getMonacoThemeByEditorTheme(theme) {
        switch (theme) {
            case EditorTheme.DARK:
                return "vs-dark";
            case EditorTheme.HIGH_CONTRAST:
                return "hc-black";
            default:
                return "vs";
        }
    }
    dispose() {
        var _a;
        (_a = this.editor) === null || _a === void 0 ? void 0 : _a.dispose();
    }
}
//# sourceMappingURL=MonacoEditorController.js.map