/*
 * Decompiled with CFR 0.152.
 */
package org.apache.knox.gateway.util;

import java.io.PrintStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.knox.gateway.i18n.messages.MessagesFactory;
import org.apache.knox.gateway.services.security.AliasService;
import org.apache.knox.gateway.services.security.AliasServiceException;
import org.apache.knox.gateway.services.security.token.TokenMetadata;
import org.apache.knox.gateway.services.security.token.TokenStateService;
import org.apache.knox.gateway.services.token.impl.TokenStateServiceMessages;
import org.apache.knox.gateway.util.Tokens;

public class TokenMigrationTool {
    private static final TokenStateServiceMessages LOG = (TokenStateServiceMessages)MessagesFactory.get(TokenStateServiceMessages.class);
    private final AliasService aliasService;
    private final TokenStateService tokenStateService;
    private final PrintStream out;
    private int progressCount = 10;
    private boolean archiveMigratedTokens;
    private boolean migrateExpiredTokens;
    private boolean verbose;

    public TokenMigrationTool(AliasService aliasService, TokenStateService tokenStateService, PrintStream out) {
        this.aliasService = aliasService;
        this.tokenStateService = tokenStateService;
        this.out = out;
    }

    public void setProgressCount(int progressCount) {
        this.progressCount = progressCount;
    }

    public void setArchiveMigratedTokens(boolean archiveMigratedTokens) {
        this.archiveMigratedTokens = archiveMigratedTokens;
    }

    public void setMigrateExpiredTokens(boolean migrateExpiredTokens) {
        this.migrateExpiredTokens = migrateExpiredTokens;
    }

    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    public void migrateTokensFromGatewayCredentialStore() {
        try {
            ConcurrentHashMap<String, TokenData> tokenDataMap = new ConcurrentHashMap<String, TokenData>();
            long start = System.currentTimeMillis();
            String logMessage = "Loading token aliases from the __gateway credential store. This could take a while.";
            this.log(logMessage);
            Map passwordAliasMap = this.aliasService.getPasswordsForGateway();
            this.log("Token aliases loaded in " + (System.currentTimeMillis() - start) + " milliseconds");
            for (Map.Entry<String, char[]> entry2 : passwordAliasMap.entrySet()) {
                String alias = (String)entry2.getKey();
                this.processAlias(passwordAliasMap, entry2, alias, tokenDataMap);
            }
            long migrationStart = System.currentTimeMillis();
            AtomicInteger count = new AtomicInteger(0);
            tokenDataMap.entrySet().forEach(entry -> {
                int loggedCount = 0;
                this.saveTokenIfComplete(this.tokenStateService, (String)entry.getKey(), (TokenData)entry.getValue());
                count.incrementAndGet();
                if (count.intValue() > 0 && count.intValue() % this.progressCount == 0 && loggedCount != count.intValue()) {
                    loggedCount = count.intValue();
                    this.logProgress(count.intValue(), System.currentTimeMillis() - migrationStart);
                }
            });
            this.logProgress(count.intValue(), System.currentTimeMillis() - migrationStart);
            this.archiveTokens(tokenDataMap);
            this.removeTokenAliasesFromGatewayCredentialStore(tokenDataMap);
        }
        catch (AliasServiceException e) {
            throw new RuntimeException("Error while migrating tokens from __gateway credential store: " + e.getMessage(), e);
        }
    }

    private void log(String message) {
        LOG.info(message);
        if (this.out != null) {
            this.out.println(message);
        }
    }

    private void logProgress(int count, long duration) {
        this.log(String.format(Locale.getDefault(), "Processed %d tokens in %d milliseconds", count, duration));
    }

    private void processAlias(Map<String, char[]> passwordAliasMap, Map.Entry<String, char[]> passwordAliasMapEntry, String alias, Map<String, TokenData> tokenDataMap) {
        String tokenId = null;
        if (alias.endsWith("--max")) {
            tokenId = alias.substring(0, alias.indexOf("--max"));
            tokenDataMap.putIfAbsent(tokenId, new TokenData());
            long expiration = this.convertCharArrayToLong(passwordAliasMap.get(tokenId));
            long maxLifeTime = this.convertCharArrayToLong(passwordAliasMapEntry.getValue());
            tokenDataMap.get((Object)tokenId).expiration = expiration;
            tokenDataMap.get((Object)tokenId).maxLifeTime = maxLifeTime;
        } else if (alias.endsWith("--meta")) {
            tokenId = alias.substring(0, alias.indexOf("--meta"));
            tokenDataMap.putIfAbsent(tokenId, new TokenData());
            tokenDataMap.get((Object)tokenId).metadata = TokenMetadata.fromJSON((String)new String(passwordAliasMapEntry.getValue()));
        } else if (alias.endsWith("--iss")) {
            tokenId = alias.substring(0, alias.indexOf("--iss"));
            tokenDataMap.putIfAbsent(tokenId, new TokenData());
            tokenDataMap.get((Object)tokenId).issueTime = this.convertCharArrayToLong(passwordAliasMapEntry.getValue());
        }
    }

    private long convertCharArrayToLong(char[] charArray) {
        return Long.parseLong(new String(charArray));
    }

    private void saveTokenIfComplete(TokenStateService tokenStateService, String tokenId, TokenData tokenData) {
        if (tokenId != null && tokenData.isComplete() && !tokenData.isProcessed()) {
            if (this.migrateToken(tokenData)) {
                tokenStateService.addToken(tokenId, tokenData.issueTime, tokenData.expiration, tokenData.maxLifeTime);
                tokenStateService.addMetadata(tokenId, tokenData.metadata);
                if (this.verbose) {
                    this.log("Migrated token " + Tokens.getTokenIDDisplayText((String)tokenId) + " into the configured TokenStateService backend.");
                }
            } else if (this.verbose) {
                this.log("Skipping the migration of expired token with ID = " + Tokens.getTokenIDDisplayText((String)tokenId));
            }
        }
        tokenData.processed = true;
    }

    private boolean migrateToken(TokenData tokenData) {
        return tokenData.isExpired() ? this.migrateExpiredTokens : true;
    }

    private void archiveTokens(Map<String, TokenData> tokenDataMap) throws AliasServiceException {
        if (this.archiveMigratedTokens) {
            String cluster = "__tokens";
            this.log("Archiving token aliases in the __tokens credential store...");
            long start = System.currentTimeMillis();
            HashMap tokenAliasesToArchive = new HashMap();
            tokenDataMap.entrySet().forEach(tokenDataMapEntry -> {
                String tokenId = (String)tokenDataMapEntry.getKey();
                tokenDataMapEntry.getValue();
                tokenAliasesToArchive.put(tokenId, String.valueOf(((TokenData)tokenDataMapEntry.getValue()).expiration));
                tokenAliasesToArchive.put(tokenId + "--max", String.valueOf(((TokenData)tokenDataMapEntry.getValue()).maxLifeTime));
                tokenAliasesToArchive.put(tokenId + "--iss", String.valueOf(((TokenData)tokenDataMapEntry.getValue()).issueTime));
                tokenAliasesToArchive.put(tokenId + "--meta", ((TokenData)tokenDataMapEntry.getValue()).metadata.toJSON());
            });
            this.aliasService.addAliasesForCluster("__tokens", tokenAliasesToArchive);
            this.log("Archived token related aliases in the __tokens credential store in " + (System.currentTimeMillis() - start) + " millsieconds ");
        }
    }

    private void removeTokenAliasesFromGatewayCredentialStore(Map<String, TokenData> tokenDataMap) throws AliasServiceException {
        this.log("Removing token aliases from the __gateway credential store...");
        long start = System.currentTimeMillis();
        HashSet tokenAliases = new HashSet();
        tokenDataMap.entrySet().forEach(tokenDataMapEntry -> {
            String tokenId = (String)tokenDataMapEntry.getKey();
            tokenAliases.addAll(Arrays.asList(tokenId, tokenId + "--max", tokenId + "--iss", tokenId + "--meta"));
        });
        this.aliasService.removeAliasesForCluster("__gateway", tokenAliases);
        this.log("Removed token related aliases from the __gateway credential store in " + (System.currentTimeMillis() - start) + " milliseconds");
    }

    private class TokenData {
        boolean processed;
        long issueTime = -1L;
        long maxLifeTime = -1L;
        long expiration = -2L;
        TokenMetadata metadata;

        private TokenData() {
        }

        boolean isComplete() {
            return this.issueTime != -1L && this.maxLifeTime != -1L && this.expiration != -2L && this.metadata != null;
        }

        boolean isProcessed() {
            return this.processed;
        }

        boolean isExpired() {
            return this.expiration == -1L ? false : this.expiration < System.currentTimeMillis();
        }
    }
}

